<?php
require_once 'config.php';

define('CANCEL_SALES_FOLDER_ID', '1Sb3NCZsqTAoqybnIowT_saavfUpTp7jn');
define('PRODUK_KHUSUS_FOLDER_ID', '1oFwOJg7fS5IPHQRALDnDl6JQDYEubx3F');
define('SO_KAS_TOKO_FOLDER_ID', '1YdmsL7XcKoKyoISahK6iGP_zAK573MiP');
define('VARIANCE_PENGGANTIAN_FOLDER_ID', '1iBmUtbEOrY-HV4ciHSCuWKtrP3--IzU4');

// PERBAIKAN: Definisikan folder ID khusus untuk TREND, jangan menggunakan yang sama dengan nkl.php
define('TREND_NBR_FOLDER_ID', '19SJhQl-AqlV__9FUB_rvZvcybXbOz3Zj');
define('TREND_NKL_FOLDER_ID', '1c__bZAtTrRchA_QOJ9b34-6Diks7j8vZ');
define('NKL_OTOMATIS_FOLDER_ID', '1KIKbSxkZlQHri59D5P02j6dlK9hBJkX3');
define('RETUR_SALES_FOLDER_ID', '1MOH-rk5rRV8P2MyKKg7n0_x1PjzwLXiL');
define('RUSAK_TIKUS_FOLDER_ID', '1xn5XMENhwG6ybsZxGeF_GVL1quDcNEWn');

function handleTREND($telegram_id, $chat_id, $text, $token_access = false) {
    if (!$token_access) {
        $user = isUserRegistered($telegram_id);
        if (!$user) {
            sendMessage($chat_id, "❌ Anda belum terdaftar dalam sistem.");
            return;
        }
    }
    
    // Set flag untuk token access jika digunakan
    if ($token_access) {
        $temp_data = getTemporaryData($telegram_id);
        $temp_data['trend_token_access'] = true;
        $temp_data['trend_token_timestamp'] = time();
        saveTemporaryData($telegram_id, $temp_data);
        error_log("Token access flag set for TREND - User: $telegram_id");
    }
    
    $message = "📈 Anda memilih fitur TREND\n\nSilakan pilih folder mana yang anda akses:";
    
    $inline_keyboard = [
        'inline_keyboard' => [
            [
                ['text' => '❌ Cancel Sales', 'callback_data' => 'trend_cancel_sales'],
                ['text' => '🎯 Produk Khusus', 'callback_data' => 'trend_produk_khusus']
            ],
            [
                ['text' => '🪙 SO Kas Toko', 'callback_data' => 'trend_so_kas_toko'],
                ['text' => '🔄 Variance Penggantian', 'callback_data' => 'trend_variance_penggantian']
            ],
            [
                ['text' => '📊 NBR', 'callback_data' => 'trend_nbr'],
                ['text' => '📁 NKL', 'callback_data' => 'trend_nkl']
            ],
            [
                ['text' => '🤖 NKL Otomatis', 'callback_data' => 'trend_nkl_otomatis'],
                ['text' => '↩️ Retur Sales', 'callback_data' => 'trend_retur_sales']
            ],
            [
                ['text' => '🐭 Rusak Tikus', 'callback_data' => 'trend_rusak_tikus']
            ]
        ]
    ];
    
    sendMessage($chat_id, $message, $inline_keyboard);
}

function handleTRENDCallback($telegram_id, $chat_id, $callback_data) {
    // Cek apakah ada akses token untuk TREND
    $temp_data = getTemporaryData($telegram_id);
    $has_token_access = false;
    
    if (isset($temp_data['trend_token_access']) && $temp_data['trend_token_access'] === true) {
        // Cek apakah token access masih valid (dalam 30 menit terakhir)
        $token_timestamp = isset($temp_data['trend_token_timestamp']) ? $temp_data['trend_token_timestamp'] : 0;
        $current_time = time();
        
        if (($current_time - $token_timestamp) <= 1800) { // 30 menit = 1800 detik
            $has_token_access = true;
            error_log("Token access validated for TREND callback - User: $telegram_id");
        } else {
            // Token access expired, hapus flag
            unset($temp_data['trend_token_access']);
            unset($temp_data['trend_token_timestamp']);
            saveTemporaryData($telegram_id, $temp_data);
            error_log("Token access expired for TREND callback - User: $telegram_id");
        }
    }
    
    // Jika tidak ada token access, cek registrasi normal
    if (!$has_token_access) {
        $user = isUserRegistered($telegram_id);
        if (!$user) {
            sendMessage($chat_id, "❌ Anda belum terdaftar dalam sistem dan tidak memiliki akses token yang valid.");
            return;
        }
    }
    
    // Cek apakah sedang ada proses TREND yang berjalan untuk user ini
    if (isset($temp_data['trend_processing']) && $temp_data['trend_processing'] === true) {
        sendMessage($chat_id, "⏳ Proses pengambilan file TREND sedang berlangsung. Mohon tunggu...");
        return;
    }
    
    $folder_id = '';
    $folder_name = '';
    
    // PERBAIKAN UTAMA: Gunakan folder ID yang benar untuk masing-masing kategori TREND
    switch ($callback_data) {
        case 'trend_cancel_sales':
            $folder_id = CANCEL_SALES_FOLDER_ID;
            $folder_name = 'Cancel Sales';
            break;
        case 'trend_produk_khusus':
            $folder_id = PRODUK_KHUSUS_FOLDER_ID;
            $folder_name = 'Produk Khusus';
            break;
        case 'trend_so_kas_toko':
            $folder_id = SO_KAS_TOKO_FOLDER_ID;
            $folder_name = 'SO Kas Toko';
            break;
        case 'trend_variance_penggantian':
            $folder_id = VARIANCE_PENGGANTIAN_FOLDER_ID;
            $folder_name = 'Variance Penggantian';
            break;
        case 'trend_nbr':
            // Gunakan TREND_NBR_FOLDER_ID, bukan NBR_FOLDER_ID dari nbr.php
            $folder_id = TREND_NBR_FOLDER_ID;
            $folder_name = 'NBR (TREND)';
            break;
        case 'trend_nkl':
            // PERBAIKAN: Gunakan TREND_NKL_FOLDER_ID, bukan NKL_FOLDER_ID dari nkl.php
            $folder_id = TREND_NKL_FOLDER_ID;
            $folder_name = 'NKL (TREND)';
            break;
        case 'trend_nkl_otomatis':
            $folder_id = NKL_OTOMATIS_FOLDER_ID;
            $folder_name = 'NKL Otomatis';
            break;
        case 'trend_retur_sales':
            $folder_id = RETUR_SALES_FOLDER_ID;
            $folder_name = 'Retur Sales';
            break;
        case 'trend_rusak_tikus':
            $folder_id = RUSAK_TIKUS_FOLDER_ID;
            $folder_name = 'Rusak Tikus';
            break;
        default:
            sendMessage($chat_id, "❌ Pilihan tidak valid.");
            return;
    }
    
    // Set flag bahwa proses TREND sedang berjalan
    $temp_data['trend_processing'] = true;
    $temp_data['trend_start_time'] = time();
    saveTemporaryData($telegram_id, $temp_data);
    
    try {
        // Log akses dengan folder ID yang sebenarnya digunakan
        $access_method = $has_token_access ? "TOKEN ACCESS" : "REGISTERED USER";
        error_log("TREND access - User: $telegram_id, Method: $access_method, Folder: $folder_name, Folder ID: $folder_id");
        
        // Ambil daftar file dari Google Drive (menggunakan fungsi dari nkl.php)
        $files = getFilesFromGoogleDrive($folder_id);
        
        if (empty($files)) {
            sendMessage($chat_id, "📂 Folder {$folder_name} kosong atau tidak dapat diakses.");
            clearTRENDProcessing($telegram_id);
            return;
        }
        
        // Pesan yang sudah dimodifikasi dengan indikasi access method
        $access_info = $has_token_access ? " (menggunakan token akses)" : "";
        sendMessage($chat_id, "📈 AIC ketemu " . count($files) . " file {$folder_name} nih{$access_info}. Aku proses dulu ya..");
        
        // Kirim setiap file
        $success_count = 0;
        $error_count = 0;
        $sent_files = [];
        
        foreach ($files as $index => $file) {
            // Add debug logging dengan folder ID
            error_log("Processing TREND file: " . $file['name'] . " (ID: " . $file['id'] . ") from folder: $folder_id - Access: $access_method");
            
            $result = sendTRENDFile($chat_id, $file);
            if ($result['success']) {
                $success_count++;
                $sent_files[] = $file['name'];
                error_log("Successfully sent TREND file: " . $file['name']);
            } else {
                $error_count++;
                error_log("Failed to send TREND file: " . $file['name'] . " - " . $result['error']);
            }
            
            // Delay untuk menghindari rate limit Telegram
            sleep(1);
            
            // Safety check - jika proses terlalu lama (lebih dari 15 menit), stop
            if (time() - $temp_data['trend_start_time'] > 900) {
                sendMessage($chat_id, "⚠️ Proses dihentikan karena timeout. Sebagian file mungkin belum terkirim.");
                break;
            }
        }
        
        // Pesan penutup yang sudah dimodifikasi dengan summary
        $summary = "📈 Udah selesai nih... ";
        if ($error_count > 0) {
            $summary .= "($success_count berhasil, $error_count gagal) ";
        }
        $summary .= "Panggil aja aku jika butuh data lagi, aku selalu siap membantu";
        
        // Tambahkan info token jika menggunakan akses token
        if ($has_token_access) {
            $summary .= "\n\n🔑 <i>Akses menggunakan token sementara.</i>";
        }
        
        sendMessage($chat_id, $summary);
        
        // Hapus token access flag setelah selesai untuk security
        if ($has_token_access) {
            unset($temp_data['trend_token_access']);
            unset($temp_data['trend_token_timestamp']);
            saveTemporaryData($telegram_id, $temp_data);
            error_log("Token access flag cleared after TREND processing - User: $telegram_id");
        }
        
    } catch (Exception $e) {
        error_log("Error in handleTRENDCallback: " . $e->getMessage());
        sendMessage($chat_id, "❌ Terjadi kesalahan saat mengambil file dari Google Drive: " . $e->getMessage());
    } finally {
        // Hapus flag processing
        clearTRENDProcessing($telegram_id);
    }
}

function clearTRENDProcessing($telegram_id) {
    $temp_data = getTemporaryData($telegram_id);
    unset($temp_data['trend_processing']);
    unset($temp_data['trend_start_time']);
    saveTemporaryData($telegram_id, $temp_data);
}

// Fungsi helper untuk membersihkan token access yang expired
function cleanupExpiredTRENDTokenAccess($telegram_id = null) {
    if ($telegram_id) {
        // Bersihkan untuk user tertentu
        $temp_data = getTemporaryData($telegram_id);
        if (isset($temp_data['trend_token_access']) && isset($temp_data['trend_token_timestamp'])) {
            $current_time = time();
            if (($current_time - $temp_data['trend_token_timestamp']) > 1800) {
                unset($temp_data['trend_token_access']);
                unset($temp_data['trend_token_timestamp']);
                saveTemporaryData($telegram_id, $temp_data);
                error_log("Cleaned up expired TREND token access for user: $telegram_id");
            }
        }
    }
}

function sendTRENDFile($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        // Add debug logging
        error_log("Attempting to send TREND file: " . $file['name']);
        
        // Cek ekstensi file
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        error_log("File extension: " . $file_extension);
        
        // Jika file adalah .txt, kirim sebagai teks biasa
        if ($file_extension === 'txt') {
            error_log("Processing as txt file");
            return sendTxtFileAsNormalMessageTREND($chat_id, $file);
        }
        
        // Untuk file non-txt, gunakan logic yang sama dengan NBR
        error_log("Processing as document file");
        return sendFileFromGoogleDrive($chat_id, $file);
        
    } catch (Exception $e) {
        error_log("Error in sendTRENDFile: " . $e->getMessage());
        return sendFileLink($chat_id, $file, $e->getMessage());
    }
}

function sendTxtFileAsNormalMessageTREND($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        error_log("Starting sendTxtFileAsNormalMessageTREND for: " . $file['name']);
        
        // Cek ukuran file untuk txt (batasi maksimal 2MB untuk txt)
        $max_txt_size = 2 * 1024 * 1024; // 2MB
        if (isset($file['size']) && $file['size'] > $max_txt_size) {
            error_log("File too large: " . $file['size'] . " bytes");
            return sendFileLink($chat_id, $file, "File txt terlalu besar untuk dikirim sebagai teks");
        }
        
        // Download file txt menggunakan fungsi yang sudah ada di nkl.php
        error_log("Downloading file content using multiple methods...");
        $txt_content = downloadFileWithMultipleMethodsTREND($file['id']);
        
        if ($txt_content === false || $txt_content === null) {
            error_log("Downloaded content is false or null");
            return sendFileLink($chat_id, $file, "Tidak dapat mengunduh file txt");
        }
        
        // Check if content is empty after download
        if (empty(trim($txt_content))) {
            error_log("Downloaded content is empty after trim");
            return sendFileLink($chat_id, $file, "File txt kosong");
        }
        
        error_log("Downloaded content length: " . strlen($txt_content));
        error_log("Content preview: " . substr($txt_content, 0, 100) . "...");
        
        // PERBAIKAN UTAMA: Bersihkan encoding sebelum memproses konten
        $txt_content = fixEncodingAndCleanContentTREND($txt_content);
        
        if (empty(trim($txt_content))) {
            error_log("Content is empty after encoding fix and cleaning");
            return sendFileLink($chat_id, $file, "File txt kosong setelah pembersihan");
        }
        
        error_log("Content after encoding fix - length: " . strlen($txt_content));
        
        // Batasi panjang pesan Telegram (max 4096 karakter)
        $max_message_length = 4000; // Sisakan ruang untuk nama file
        
        $file_name = htmlspecialchars($file['name'], ENT_QUOTES, 'UTF-8');
        $message_header = "📄 <b>{$file_name}</b>\n\n";
        
        if (strlen($txt_content) > ($max_message_length - strlen($message_header))) {
            error_log("Content too long, splitting into chunks");
            // Jika terlalu panjang, kirim dalam beberapa bagian
            return sendLongNormalTextContentTREND($chat_id, $file_name, $txt_content, $message_header);
        } else {
            // Kirim sebagai teks biasa tanpa format kode
            $full_message = $message_header . $txt_content;
            error_log("Sending single message with length: " . strlen($full_message));
            
            // Pastikan message juga valid UTF-8
            if (!mb_check_encoding($full_message, 'UTF-8')) {
                error_log("Final message is not valid UTF-8, attempting to fix");
                $full_message = mb_convert_encoding($full_message, 'UTF-8', 'UTF-8');
            }
            
            $send_result = sendMessage($chat_id, $full_message);
            
            if ($send_result && isset($send_result['ok']) && $send_result['ok']) {
                $result['success'] = true;
                error_log("Successfully sent txt content as normal message");
            } else {
                error_log("Failed to send message. Result: " . print_r($send_result, true));
                return sendFileLink($chat_id, $file, "Gagal mengirim konten txt sebagai pesan");
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Exception in sendTxtFileAsNormalMessageTREND: " . $e->getMessage());
        return sendFileLink($chat_id, $file, "Error saat memproses file txt: " . $e->getMessage());
    }
}

// FUNGSI BARU: Perbaikan encoding dan pembersihan konten untuk TREND
function fixEncodingAndCleanContentTREND($content) {
    error_log("Starting encoding fix and content cleaning for TREND");
    
    // Step 1: Deteksi encoding asli
    $detected_encoding = mb_detect_encoding($content, ['UTF-8', 'Windows-1252', 'ISO-8859-1', 'ASCII'], true);
    error_log("Detected encoding: " . ($detected_encoding ?: 'unknown'));
    
    // Step 2: Convert ke UTF-8 jika perlu
    if ($detected_encoding && $detected_encoding !== 'UTF-8') {
        $content = mb_convert_encoding($content, 'UTF-8', $detected_encoding);
        error_log("Converted from $detected_encoding to UTF-8");
    } else if (!$detected_encoding || !mb_check_encoding($content, 'UTF-8')) {
        // Jika encoding tidak terdeteksi atau tidak valid UTF-8, coba beberapa pendekatan
        error_log("Encoding detection failed or invalid UTF-8, trying alternative approaches");
        
        // Coba convert dari Windows-1252 (encoding yang sering bermasalah)
        $content = @mb_convert_encoding($content, 'UTF-8', 'Windows-1252');
        
        // Jika masih tidak valid, gunakan iconv dengan ignore
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @iconv('UTF-8', 'UTF-8//IGNORE', $content);
        }
        
        // Sebagai langkah terakhir, hapus karakter non-UTF-8
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @mb_convert_encoding($content, 'UTF-8', 'UTF-8');
        }
    }
    
    // Step 3: Bersihkan karakter kontrol dan karakter bermasalah
    $content = preg_replace('/\r\n|\r/', "\n", $content); // Normalize line endings
    $content = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $content); // Remove control characters
    
    // Step 4: Bersihkan karakter invisible dan karakter Unicode bermasalah
    $content = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $content); // Remove zero-width chars
    $content = preg_replace('/[\x{00A0}]/u', ' ', $content); // Replace non-breaking space with regular space
    
    // Step 5: Trim dan bersihkan whitespace berlebihan
    $content = trim($content);
    $content = preg_replace('/\n{3,}/', "\n\n", $content); // Replace multiple newlines with max 2
    $content = preg_replace('/[ \t]+/', ' ', $content); // Replace multiple spaces/tabs with single space
    
    // Step 6: Validasi final
    if (!mb_check_encoding($content, 'UTF-8')) {
        error_log("Content still not valid UTF-8 after all fixes, using fallback");
        // Sebagai fallback terakhir, encode semua karakter non-ASCII
        $content = preg_replace_callback('/[^\x20-\x7E\n]/', function($matches) {
            return '?'; // Replace dengan question mark
        }, $content);
    }
    
    error_log("Encoding fix completed, final content length: " . strlen($content));
    error_log("Is valid UTF-8: " . (mb_check_encoding($content, 'UTF-8') ? 'yes' : 'no'));
    
    return $content;
}

// Fungsi untuk download dengan multiple methods (menggunakan fungsi dari nkl.php) untuk TREND
function downloadFileWithMultipleMethodsTREND($file_id) {
    error_log("Attempting to download TREND file with ID: $file_id");
    
    // Coba method 1 dari nkl.php
    $file_content = downloadFileMethod1($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded TREND file with method1, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method1 failed for TREND file, trying method3");
    
    // Coba method 3 dari nkl.php
    $file_content = downloadFileMethod3($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded TREND file with method3, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method3 failed for TREND file, trying direct URL");
    
    // Coba direct URL approach
    $file_content = downloadDirectUrl("https://drive.google.com/uc?export=download&id=" . $file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded TREND file with direct URL, size: " . strlen($file_content));
        return $file_content;
    }
    
    error_log("All download methods failed for TREND file ID: $file_id");
    return false;
}

function sendLongNormalTextContentTREND($chat_id, $file_name, $txt_content, $message_header) {
    $max_chunk_size = 3900; // Batasi per chunk
    $chunks = str_split($txt_content, $max_chunk_size);
    $total_chunks = count($chunks);
    
    $result = ['success' => true, 'error' => ''];
    
    error_log("Splitting TREND content into $total_chunks chunks");
    
    foreach ($chunks as $index => $chunk) {
        $chunk_number = $index + 1;
        $chunk_header = $message_header;
        
        if ($total_chunks > 1) {
            $chunk_header .= "📋 <b>Bagian {$chunk_number}/{$total_chunks}</b>\n\n";
        }
        
        // Kirim sebagai teks biasa tanpa tag <pre>
        $chunk_message = $chunk_header . $chunk;
        
        // Pastikan setiap chunk juga valid UTF-8
        if (!mb_check_encoding($chunk_message, 'UTF-8')) {
            $chunk_message = mb_convert_encoding($chunk_message, 'UTF-8', 'UTF-8');
        }
        
        error_log("Sending TREND chunk $chunk_number/$total_chunks with length: " . strlen($chunk_message));
        $send_result = sendMessage($chat_id, $chunk_message);
        
        if (!$send_result || !isset($send_result['ok']) || !$send_result['ok']) {
            $result['success'] = false;
            $result['error'] = "Gagal mengirim bagian {$chunk_number}";
            error_log("Failed to send TREND chunk $chunk_number. Result: " . print_r($send_result, true));
            break;
        }
        
        // Delay antar chunk
        if ($index < count($chunks) - 1) {
            sleep(1);
        }
    }
    
    return $result;
}

// Note: Fungsi-fungsi berikut sudah ada di nkl.php dan akan digunakan:
// - downloadFileMethod1()
// - downloadFileMethod3() 
// - downloadDirectUrl()
// - sendFileFromGoogleDrive()
// - sendFileLink()
// - getFilesFromGoogleDrive()

?>