<?php
// nkl.php - Fixed version with modified messages
require_once 'config.php';

// ID folder NKL di Google Drive - ganti dengan ID folder NKL Anda
define('NKL_FOLDER_ID', '1nbeIetG74-ypFCSOKTgNK-hXAwwYf1-2');
define('GOOGLE_API_KEY', 'AIzaSyAg3Y8yLuEd7A75Jv0OLJl-uzO4wzS-ozY');

// Tambahan: Service Account credentials (opsional, lebih aman)
define('SERVICE_ACCOUNT_EMAIL', ''); // Email service account jika menggunakan
define('PRIVATE_KEY_FILE', ''); // Path ke file private key jika menggunakan service account

function handleNKL($telegram_id, $chat_id, $text, $token_access = false) {
    // Jika bukan token access, cek apakah user terdaftar
    if (!$token_access) {
        $user = isUserRegistered($telegram_id);
        if (!$user) {
            sendMessage($chat_id, "⛔ Anda belum terdaftar dalam sistem.");
            return;
        }
    }
    
    // Cek apakah sedang ada proses NKL yang berjalan untuk user ini
    $temp_data = getTemporaryData($telegram_id);
    if (isset($temp_data['nkl_processing']) && $temp_data['nkl_processing'] === true) {
        sendMessage($chat_id, "⏳ Proses pengambilan file NKL sedang berlangsung. Mohon tunggu...");
        return;
    }
    
    // Set flag bahwa proses NKL sedang berjalan
    $temp_data['nkl_processing'] = true;
    $temp_data['nkl_start_time'] = time();
    saveTemporaryData($telegram_id, $temp_data);
    
    try {
        // Ambil daftar file dari Google Drive
        $files = getFilesFromGoogleDrive(NKL_FOLDER_ID);
        
        if (empty($files)) {
            sendMessage($chat_id, "📂 Folder NKL kosong atau tidak dapat diakses.");
            clearNKLProcessing($telegram_id);
            return;
        }
        
        // Pesan yang sudah dimodifikasi
        sendMessage($chat_id, "AIC ketemu " . count($files) . " File NKL nih. Aku proses dulu ya... ");
        
        // Kirim setiap file
        $success_count = 0;
        $error_count = 0;
        $sent_files = [];
        
        foreach ($files as $index => $file) {
            $result = sendFileFromGoogleDrive($chat_id, $file);
            if ($result['success']) {
                $success_count++;
                $sent_files[] = $file['name'];
            } else {
                $error_count++;
                error_log("Failed to send file: " . $file['name'] . " - " . $result['error']);
            }
            
            // Delay untuk menghindari rate limit Telegram
            sleep(1);
            
            // Safety check - jika proses terlalu lama (lebih dari 15 menit), stop
            if (time() - $temp_data['nkl_start_time'] > 900) {
                sendMessage($chat_id, "⚠️ Proses dihentikan karena timeout. Sebagian file mungkin belum terkirim.");
                break;
            }
        }
        
        // Pesan penutup yang sudah dimodifikasi
        sendMessage($chat_id, "Udah selesai nih... Panggil aja aku jika butuh data lagi, aku selalu siap membantu");
        
    } catch (Exception $e) {
        error_log("Error in handleNKL: " . $e->getMessage());
        sendMessage($chat_id, "⛔ Terjadi kesalahan saat mengambil file dari Google Drive: " . $e->getMessage());
    } finally {
        // Hapus flag processing
        clearNKLProcessing($telegram_id);
    }
}

function clearNKLProcessing($telegram_id) {
    $temp_data = getTemporaryData($telegram_id);
    unset($temp_data['nkl_processing']);
    unset($temp_data['nkl_start_time']);
    saveTemporaryData($telegram_id, $temp_data);
}

function getFilesFromGoogleDrive($folder_id) {
    $url = "https://www.googleapis.com/drive/v3/files";
    $params = [
        'key' => GOOGLE_API_KEY,
        'q' => "'{$folder_id}' in parents and trashed=false",
        'fields' => 'files(id,name,mimeType,size,webContentLink,webViewLink,permissions)',
        'pageSize' => 100,
        'orderBy' => 'name'
    ];
    
    $url .= '?' . http_build_query($params);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'User-Agent: TelegramBot/1.0'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_error($ch)) {
        curl_close($ch);
        throw new Exception("cURL Error: " . curl_error($ch));
    }
    
    curl_close($ch);
    
    if ($http_code !== 200) {
        throw new Exception("Google Drive API Error: HTTP {$http_code} - Response: " . substr($response, 0, 200));
    }
    
    $data = json_decode($response, true);
    
    if (!isset($data['files'])) {
        throw new Exception("Invalid response from Google Drive API: " . substr($response, 0, 200));
    }
    
    return $data['files'];
}

function sendFileFromGoogleDrive($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        // Cek ukuran file (maksimal 50MB untuk Telegram)
        $max_size = 50 * 1024 * 1024; // 50MB
        if (isset($file['size']) && $file['size'] > $max_size) {
            return sendLargeFileLink($chat_id, $file);
        }
        
        // Coba beberapa metode untuk mengunduh file
        $download_success = false;
        $file_content = null;
        $attempts = 0;
        $max_attempts = 3;
        
        while (!$download_success && $attempts < $max_attempts) {
            $attempts++;
            
            // Method 1: Direct download dengan API key
            if (!$download_success) {
                $file_content = downloadFileMethod1($file['id']);
                if ($file_content !== false) {
                    $download_success = true;
                    error_log("File {$file['name']} downloaded using method 1 (attempt {$attempts})");
                }
            }
            
            // Method 2: Export jika file adalah Google Workspace document
            if (!$download_success && isGoogleWorkspaceFile($file['mimeType'])) {
                $file_content = exportGoogleWorkspaceFile($file['id'], $file['mimeType']);
                if ($file_content !== false) {
                    $download_success = true;
                    $file['name'] = addExportExtension($file['name'], $file['mimeType']);
                    error_log("File {$file['name']} exported using method 2 (attempt {$attempts})");
                }
            }
            
            // Method 3: Coba dengan different user agent dan headers
            if (!$download_success) {
                $file_content = downloadFileMethod3($file['id']);
                if ($file_content !== false) {
                    $download_success = true;
                    error_log("File {$file['name']} downloaded using method 3 (attempt {$attempts})");
                }
            }
            
            if (!$download_success && $attempts < $max_attempts) {
                sleep(2); // Wait before retry
            }
        }
        
        if (!$download_success || !$file_content) {
            // Jika semua method gagal, kirim link saja
            return sendFileLink($chat_id, $file, "Tidak dapat mengunduh file secara langsung");
        }
        
        // Simpan file sementara dengan nama yang aman
        $safe_filename = sanitizeFilename($file['name']);
        $temp_file = sys_get_temp_dir() . '/' . uniqid('nkl_') . '_' . $safe_filename;
        
        if (!file_put_contents($temp_file, $file_content)) {
            throw new Exception("Failed to save temporary file");
        }
        
        // Kirim sebagai document
        $sent = sendDocument($chat_id, $temp_file, $file['name'], $file);
        
        // Hapus file sementara
        if (file_exists($temp_file)) {
            unlink($temp_file);
        }
        
        if ($sent) {
            $result['success'] = true;
        } else {
            // Jika gagal kirim dokumen, coba kirim sebagai link
            return sendFileLink($chat_id, $file, "Gagal mengirim file sebagai dokumen");
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Error in sendFileFromGoogleDrive: " . $e->getMessage());
        return sendFileLink($chat_id, $file, $e->getMessage());
    }
}

function downloadFileMethod1($file_id) {
    $file_url = "https://www.googleapis.com/drive/v3/files/{$file_id}?alt=media&key=" . GOOGLE_API_KEY;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $file_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: */*',
        'Accept-Encoding: identity'
    ]);
    
    $file_content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $file_content && strlen($file_content) > 0) {
        return $file_content;
    }
    
    return false;
}

function downloadFileMethod3($file_id) {
    // Coba dengan direct download tanpa API key menggunakan different approach
    $file_url = "https://drive.google.com/uc?export=download&id={$file_id}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $file_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: en-US,en;q=0.5',
        'Accept-Encoding: identity',
        'Connection: keep-alive'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $response && strlen($response) > 0) {
        // Cek apakah response adalah HTML (virus scan warning)
        if (strpos($response, '<html') !== false && strpos($response, 'virus scan') !== false) {
            // Extract download link dari virus scan page
            if (preg_match('/href="([^"]+export=download[^"]+)"/', $response, $matches)) {
                $download_url = html_entity_decode($matches[1]);
                return downloadDirectUrl($download_url);
            }
        } else {
            return $response;
        }
    }
    
    return false;
}

function downloadDirectUrl($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $file_content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $file_content && strlen($file_content) > 0) {
        return $file_content;
    }
    
    return false;
}

function isGoogleWorkspaceFile($mimeType) {
    $workspace_types = [
        'application/vnd.google-apps.document',
        'application/vnd.google-apps.spreadsheet',
        'application/vnd.google-apps.presentation',
        'application/vnd.google-apps.drawing'
    ];
    
    return in_array($mimeType, $workspace_types);
}

function exportGoogleWorkspaceFile($file_id, $mimeType) {
    $export_formats = [
        'application/vnd.google-apps.document' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'application/vnd.google-apps.spreadsheet' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'application/vnd.google-apps.presentation' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation'
    ];
    
    if (!isset($export_formats[$mimeType])) {
        return false;
    }
    
    $export_mime = $export_formats[$mimeType];
    $export_url = "https://www.googleapis.com/drive/v3/files/{$file_id}/export?mimeType=" . urlencode($export_mime) . "&key=" . GOOGLE_API_KEY;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $export_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; TelegramBot/1.0)');
    
    $file_content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code === 200 && $file_content && strlen($file_content) > 0) {
        return $file_content;
    }
    
    return false;
}

function addExportExtension($filename, $mimeType) {
    $extensions = [
        'application/vnd.google-apps.document' => '.docx',
        'application/vnd.google-apps.spreadsheet' => '.xlsx',
        'application/vnd.google-apps.presentation' => '.pptx'
    ];
    
    if (isset($extensions[$mimeType])) {
        $pathinfo = pathinfo($filename);
        return $pathinfo['filename'] . $extensions[$mimeType];
    }
    
    return $filename;
}

function sanitizeFilename($filename) {
    // Remove atau replace karakter yang tidak diperbolehkan
    $filename = preg_replace('/[^\w\s\.-]/', '_', $filename);
    $filename = preg_replace('/\s+/', '_', $filename);
    return substr($filename, 0, 100); // Batasi panjang nama file
}

function sendLargeFileLink($chat_id, $file) {
    $message = "📎 <b>" . htmlspecialchars($file['name']) . "</b>\n";
    $message .= "⚠️ File terlalu besar untuk dikirim langsung\n";
    $message .= "🔗 <a href=\"{$file['webViewLink']}\">Klik di sini untuk mengunduh</a>";
    
    $send_result = sendMessage($chat_id, $message);
    return [
        'success' => $send_result && isset($send_result['ok']) && $send_result['ok'],
        'error' => $send_result ? '' : 'Failed to send large file link'
    ];
}

function sendFileLink($chat_id, $file, $error_reason = '') {
    $message = "📎 <b>" . htmlspecialchars($file['name']) . "</b>\n";
    if ($error_reason) {
        $message .= "⚠️ {$error_reason}\n";
    }
    $message .= "🔗 <a href=\"{$file['webViewLink']}\">Klik di sini untuk mengunduh</a>";
    
    $send_result = sendMessage($chat_id, $message);
    return [
        'success' => $send_result && isset($send_result['ok']) && $send_result['ok'],
        'error' => $error_reason ?: 'Sent as link'
    ];
}

function sendDocument($chat_id, $file_path, $file_name, $file_info) {
    $url = API_URL . 'sendDocument';
    
    // Buat caption yang aman tanpa ukuran file
    $safe_name = htmlspecialchars($file_name);
    $caption = "📄 <b>{$safe_name}</b>";
    
    // Batasi panjang caption (max 1024 karakter untuk Telegram)
    if (strlen($caption) > 1000) {
        $caption = substr($caption, 0, 997) . '...';
    }
    
    $mime_type = isset($file_info['mimeType']) ? $file_info['mimeType'] : 'application/octet-stream';
    
    $data = [
        'chat_id' => $chat_id,
        'document' => new CURLFile($file_path, $mime_type, $file_name),
        'caption' => $caption,
        'parse_mode' => 'HTML'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 120);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response === false) {
        error_log("Failed to send document: cURL error");
        return false;
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        error_log("Failed to decode Telegram response: " . substr($response, 0, 200));
        return false;
    }
    
    if (!isset($result['ok']) || $result['ok'] !== true) {
        $error_msg = isset($result['description']) ? $result['description'] : 'Unknown error';
        error_log("Telegram API error: " . $error_msg);
        return false;
    }
    
    return true;
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>