<?php
// detail-nkl.php - Handler untuk Detail NKL dengan penanganan khusus file txt
require_once 'config.php';

// ID folder Detail NKL di Google Drive - ganti dengan ID folder Detail NKL Anda
define('DETAIL_NKL_FOLDER_ID', '1xw3zJlVdu_wT1dZw0mZNk9MI4aGN4LYa');

function handleDetailNKL($telegram_id, $chat_id, $text, $token_access = false) {
    // Jika bukan token access, cek apakah user terdaftar
    if (!$token_access) {
        $user = isUserRegistered($telegram_id);
        if (!$user) {
            sendMessage($chat_id, "❌ Anda belum terdaftar dalam sistem.");
            return;
        }
    }
    
    // Cek apakah sedang ada proses Detail NKL yang berjalan untuk user ini
    $temp_data = getTemporaryData($telegram_id);
    if (isset($temp_data['detail_nkl_processing']) && $temp_data['detail_nkl_processing'] === true) {
        sendMessage($chat_id, "⏳ Proses pengambilan file Detail NKL sedang berlangsung. Mohon tunggu...");
        return;
    }
    
    // Set flag bahwa proses Detail NKL sedang berjalan
    $temp_data['detail_nkl_processing'] = true;
    $temp_data['detail_nkl_start_time'] = time();
    saveTemporaryData($telegram_id, $temp_data);
    
    try {
        // Ambil daftar file dari Google Drive (menggunakan fungsi dari nkl.php)
        $files = getFilesFromGoogleDrive(DETAIL_NKL_FOLDER_ID);
        
        if (empty($files)) {
            sendMessage($chat_id, "📂 Folder Detail NKL kosong atau tidak dapat diakses.");
            clearDetailNKLProcessing($telegram_id);
            return;
        }
        
        // Pesan yang sudah dimodifikasi
        sendMessage($chat_id, "AIC ketemu " . count($files) . " file Detail NKL nih. Sabar yaa .... Aku proses dulu");
        
        // Kirim setiap file
        $success_count = 0;
        $error_count = 0;
        $sent_files = [];
        
        foreach ($files as $index => $file) {
            // Add debug logging
            error_log("Processing Detail NKL file: " . $file['name'] . " (ID: " . $file['id'] . ")");
            
            $result = sendDetailNKLFile($chat_id, $file);
            if ($result['success']) {
                $success_count++;
                $sent_files[] = $file['name'];
                error_log("Successfully sent Detail NKL file: " . $file['name']);
            } else {
                $error_count++;
                error_log("Failed to send Detail NKL file: " . $file['name'] . " - " . $result['error']);
            }
            
            // Delay untuk menghindari rate limit Telegram
            sleep(1);
            
            // Safety check - jika proses terlalu lama (lebih dari 15 menit), stop
            if (time() - $temp_data['detail_nkl_start_time'] > 900) {
                sendMessage($chat_id, "⚠️ Proses dihentikan karena timeout. Sebagian file mungkin belum terkirim.");
                break;
            }
        }
        
        // Pesan penutup yang sudah dimodifikasi dengan summary
        $summary = "Udah selesai nih... ";
        if ($error_count > 0) {
            $summary .= "($success_count berhasil, $error_count gagal) ";
        }
        $summary .= "Panggil aja aku jika butuh data lagi yaa";
        sendMessage($chat_id, $summary);
        
    } catch (Exception $e) {
        error_log("Error in handleDetailNKL: " . $e->getMessage());
        sendMessage($chat_id, "❌ Terjadi kesalahan saat mengambil file dari Google Drive: " . $e->getMessage());
    } finally {
        // Hapus flag processing
        clearDetailNKLProcessing($telegram_id);
    }
}

function clearDetailNKLProcessing($telegram_id) {
    $temp_data = getTemporaryData($telegram_id);
    unset($temp_data['detail_nkl_processing']);
    unset($temp_data['detail_nkl_start_time']);
    saveTemporaryData($telegram_id, $temp_data);
}

function sendDetailNKLFile($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        // Add debug logging
        error_log("Attempting to send Detail NKL file: " . $file['name']);
        
        // Cek ekstensi file
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        error_log("File extension: " . $file_extension);
        
        // Jika file adalah .txt, kirim sebagai teks biasa
        if ($file_extension === 'txt') {
            error_log("Processing as txt file");
            return sendTxtFileAsNormalMessage($chat_id, $file);
        }
        
        // Untuk file non-txt, gunakan logic yang sama dengan NKL
        error_log("Processing as document file");
        return sendFileFromGoogleDrive($chat_id, $file);
        
    } catch (Exception $e) {
        error_log("Error in sendDetailNKLFile: " . $e->getMessage());
        return sendFileLink($chat_id, $file, $e->getMessage());
    }
}

function sendTxtFileAsNormalMessage($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        error_log("Starting sendTxtFileAsNormalMessage for: " . $file['name']);
        
        // Cek ukuran file untuk txt (batasi maksimal 2MB untuk txt)
        $max_txt_size = 2 * 1024 * 1024; // 2MB
        if (isset($file['size']) && $file['size'] > $max_txt_size) {
            error_log("File too large: " . $file['size'] . " bytes");
            return sendFileLink($chat_id, $file, "File txt terlalu besar untuk dikirim sebagai teks");
        }
        
        // Download file txt menggunakan fungsi yang sudah ada di nkl.php
        error_log("Downloading file content using multiple methods...");
        $txt_content = downloadFileWithMultipleMethods($file['id']);
        
        if ($txt_content === false || $txt_content === null) {
            error_log("Downloaded content is false or null");
            return sendFileLink($chat_id, $file, "Tidak dapat mengunduh file txt");
        }
        
        // Check if content is empty after download
        if (empty(trim($txt_content))) {
            error_log("Downloaded content is empty after trim");
            return sendFileLink($chat_id, $file, "File txt kosong");
        }
        
        error_log("Downloaded content length: " . strlen($txt_content));
        error_log("Content preview: " . substr($txt_content, 0, 100) . "...");
        
        // PERBAIKAN UTAMA: Bersihkan encoding sebelum memproses konten
        $txt_content = fixEncodingAndCleanContent($txt_content);
        
        if (empty(trim($txt_content))) {
            error_log("Content is empty after encoding fix and cleaning");
            return sendFileLink($chat_id, $file, "File txt kosong setelah pembersihan");
        }
        
        error_log("Content after encoding fix - length: " . strlen($txt_content));
        
        // Batasi panjang pesan Telegram (max 4096 karakter)
        $max_message_length = 4000; // Sisakan ruang untuk nama file
        
        $file_name = htmlspecialchars($file['name'], ENT_QUOTES, 'UTF-8');
        $message_header = "📄 <b>{$file_name}</b>\n\n";
        
        if (strlen($txt_content) > ($max_message_length - strlen($message_header))) {
            error_log("Content too long, splitting into chunks");
            // Jika terlalu panjang, kirim dalam beberapa bagian
            return sendLongNormalTextContent($chat_id, $file_name, $txt_content, $message_header);
        } else {
            // Kirim sebagai teks biasa tanpa format kode
            $full_message = $message_header . $txt_content;
            error_log("Sending single message with length: " . strlen($full_message));
            
            // Pastikan message juga valid UTF-8
            if (!mb_check_encoding($full_message, 'UTF-8')) {
                error_log("Final message is not valid UTF-8, attempting to fix");
                $full_message = mb_convert_encoding($full_message, 'UTF-8', 'UTF-8');
            }
            
            $send_result = sendMessage($chat_id, $full_message);
            
            if ($send_result && isset($send_result['ok']) && $send_result['ok']) {
                $result['success'] = true;
                error_log("Successfully sent txt content as normal message");
            } else {
                error_log("Failed to send message. Result: " . print_r($send_result, true));
                return sendFileLink($chat_id, $file, "Gagal mengirim konten txt sebagai pesan");
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Exception in sendTxtFileAsNormalMessage: " . $e->getMessage());
        return sendFileLink($chat_id, $file, "Error saat memproses file txt: " . $e->getMessage());
    }
}

// FUNGSI BARU: Perbaikan encoding dan pembersihan konten
function fixEncodingAndCleanContent($content) {
    error_log("Starting encoding fix and content cleaning");
    
    // Step 1: Deteksi encoding asli
    $detected_encoding = mb_detect_encoding($content, ['UTF-8', 'Windows-1252', 'ISO-8859-1', 'ASCII'], true);
    error_log("Detected encoding: " . ($detected_encoding ?: 'unknown'));
    
    // Step 2: Convert ke UTF-8 jika perlu
    if ($detected_encoding && $detected_encoding !== 'UTF-8') {
        $content = mb_convert_encoding($content, 'UTF-8', $detected_encoding);
        error_log("Converted from $detected_encoding to UTF-8");
    } else if (!$detected_encoding || !mb_check_encoding($content, 'UTF-8')) {
        // Jika encoding tidak terdeteksi atau tidak valid UTF-8, coba beberapa pendekatan
        error_log("Encoding detection failed or invalid UTF-8, trying alternative approaches");
        
        // Coba convert dari Windows-1252 (encoding yang sering bermasalah)
        $content = @mb_convert_encoding($content, 'UTF-8', 'Windows-1252');
        
        // Jika masih tidak valid, gunakan iconv dengan ignore
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @iconv('UTF-8', 'UTF-8//IGNORE', $content);
        }
        
        // Sebagai langkah terakhir, hapus karakter non-UTF-8
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @mb_convert_encoding($content, 'UTF-8', 'UTF-8');
        }
    }
    
    // Step 3: Bersihkan karakter kontrol dan karakter bermasalah
    $content = preg_replace('/\r\n|\r/', "\n", $content); // Normalize line endings
    $content = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $content); // Remove control characters
    
    // Step 4: Bersihkan karakter invisible dan karakter Unicode bermasalah
    $content = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $content); // Remove zero-width chars
    $content = preg_replace('/[\x{00A0}]/u', ' ', $content); // Replace non-breaking space with regular space
    
    // Step 5: Trim dan bersihkan whitespace berlebihan
    $content = trim($content);
    $content = preg_replace('/\n{3,}/', "\n\n", $content); // Replace multiple newlines with max 2
    $content = preg_replace('/[ \t]+/', ' ', $content); // Replace multiple spaces/tabs with single space
    
    // Step 6: Validasi final
    if (!mb_check_encoding($content, 'UTF-8')) {
        error_log("Content still not valid UTF-8 after all fixes, using fallback");
        // Sebagai fallback terakhir, encode semua karakter non-ASCII
        $content = preg_replace_callback('/[^\x20-\x7E\n]/', function($matches) {
            return '?'; // Replace dengan question mark
        }, $content);
    }
    
    error_log("Encoding fix completed, final content length: " . strlen($content));
    error_log("Is valid UTF-8: " . (mb_check_encoding($content, 'UTF-8') ? 'yes' : 'no'));
    
    return $content;
}

// Fungsi untuk download dengan multiple methods (menggunakan fungsi dari nkl.php)
function downloadFileWithMultipleMethods($file_id) {
    error_log("Attempting to download file with ID: $file_id");
    
    // Coba method 1 dari nkl.php
    $file_content = downloadFileMethod1($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded with method1, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method1 failed, trying method3");
    
    // Coba method 3 dari nkl.php
    $file_content = downloadFileMethod3($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded with method3, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method3 failed, trying direct URL");
    
    // Coba direct URL approach
    $file_content = downloadDirectUrl("https://drive.google.com/uc?export=download&id=" . $file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded with direct URL, size: " . strlen($file_content));
        return $file_content;
    }
    
    error_log("All download methods failed for file ID: $file_id");
    return false;
}

function sendLongNormalTextContent($chat_id, $file_name, $txt_content, $message_header) {
    $max_chunk_size = 3900; // Batasi per chunk
    $chunks = str_split($txt_content, $max_chunk_size);
    $total_chunks = count($chunks);
    
    $result = ['success' => true, 'error' => ''];
    
    error_log("Splitting content into $total_chunks chunks");
    
    foreach ($chunks as $index => $chunk) {
        $chunk_number = $index + 1;
        $chunk_header = $message_header;
        
        if ($total_chunks > 1) {
            $chunk_header .= "📋 <b>Bagian {$chunk_number}/{$total_chunks}</b>\n\n";
        }
        
        // Kirim sebagai teks biasa tanpa tag <pre>
        $chunk_message = $chunk_header . $chunk;
        
        // Pastikan setiap chunk juga valid UTF-8
        if (!mb_check_encoding($chunk_message, 'UTF-8')) {
            $chunk_message = mb_convert_encoding($chunk_message, 'UTF-8', 'UTF-8');
        }
        
        error_log("Sending chunk $chunk_number/$total_chunks with length: " . strlen($chunk_message));
        $send_result = sendMessage($chat_id, $chunk_message);
        
        if (!$send_result || !isset($send_result['ok']) || !$send_result['ok']) {
            $result['success'] = false;
            $result['error'] = "Gagal mengirim bagian {$chunk_number}";
            error_log("Failed to send chunk $chunk_number. Result: " . print_r($send_result, true));
            break;
        }
        
        // Delay antar chunk
        if ($index < count($chunks) - 1) {
            sleep(1);
        }
    }
    
    return $result;
}

// Note: Fungsi-fungsi berikut sudah ada di nkl.php dan akan digunakan:
// - downloadFileMethod1()
// - downloadFileMethod3() 
// - downloadDirectUrl()
// - sendFileFromGoogleDrive()
// - sendFileLink()
// - getFilesFromGoogleDrive()

?>