<?php
require_once 'config.php';

define('DETAIL_NBR_FOLDER_ID', '1wB_7UL7l2ZOv3_2rc69UKHbUGks1rzYT');

function handleDetailNBR($telegram_id, $chat_id, $text, $token_access = false) {
    if (!$token_access) {
        $user = isUserRegistered($telegram_id);
        if (!$user) {
            sendMessage($chat_id, "❌ Anda belum terdaftar dalam sistem.");
            return;
        }
    }
    
    $temp_data = getTemporaryData($telegram_id);
    if (isset($temp_data['detail_nbr_processing']) && $temp_data['detail_nbr_processing'] === true) {
        sendMessage($chat_id, "⏳ Proses pengambilan file Detail NBR sedang berlangsung. Mohon tunggu...");
        return;
    }
    
    $temp_data['detail_nbr_processing'] = true;
    $temp_data['detail_nbr_start_time'] = time();
    saveTemporaryData($telegram_id, $temp_data);
    
    try {
        $files = getFilesFromGoogleDrive(DETAIL_NBR_FOLDER_ID);
        
        if (empty($files)) {
            sendMessage($chat_id, "📂 Folder Detail NBR kosong atau tidak dapat diakses.");
            clearDetailNBRProcessing($telegram_id);
            return;
        }
        
        sendMessage($chat_id, "AIC ketemu " . count($files) . " file Detail NBR nih. Sabar ya.. Aku proses dulu");
        
        $success_count = 0;
        $error_count = 0;
        $sent_files = [];
        
        foreach ($files as $index => $file) {
            error_log("Processing Detail NBR file: " . $file['name'] . " (ID: " . $file['id'] . ")");
            
            $result = sendDetailNBRFile($chat_id, $file);
            if ($result['success']) {
                $success_count++;
                $sent_files[] = $file['name'];
                error_log("Successfully sent Detail NBR file: " . $file['name']);
            } else {
                $error_count++;
                error_log("Failed to send Detail NBR file: " . $file['name'] . " - " . $result['error']);
            }
            
            sleep(1);
            
            if (time() - $temp_data['detail_nbr_start_time'] > 900) {
                sendMessage($chat_id, "⚠️ Proses dihentikan karena timeout. Sebagian file mungkin belum terkirim.");
                break;
            }
        }
        
        $summary = "Udah selesai nih... ";
        if ($error_count > 0) {
            $summary .= "($success_count berhasil, $error_count gagal) ";
        }
        $summary .= "Panggil aja aku jika butuh data lagi ya..";
        sendMessage($chat_id, $summary);
        
    } catch (Exception $e) {
        error_log("Error in handleDetailNBR: " . $e->getMessage());
        sendMessage($chat_id, "❌ Terjadi kesalahan saat mengambil file dari Google Drive: " . $e->getMessage());
    } finally {
        clearDetailNBRProcessing($telegram_id);
    }
}

function clearDetailNBRProcessing($telegram_id) {
    $temp_data = getTemporaryData($telegram_id);
    unset($temp_data['detail_nbr_processing']);
    unset($temp_data['detail_nbr_start_time']);
    saveTemporaryData($telegram_id, $temp_data);
}

function sendDetailNBRFile($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        error_log("Attempting to send Detail NBR file: " . $file['name']);
        
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        error_log("File extension: " . $file_extension);
        
        if ($file_extension === 'txt') {
            error_log("Processing as txt file");
            return sendTxtFileAsNormalMessageNBR($chat_id, $file);
        }
        
        error_log("Processing as document file");
        return sendFileFromGoogleDrive($chat_id, $file);
        
    } catch (Exception $e) {
        error_log("Error in sendDetailNBRFile: " . $e->getMessage());
        return sendFileLink($chat_id, $file, $e->getMessage());
    }
}

function sendTxtFileAsNormalMessageNBR($chat_id, $file) {
    $result = ['success' => false, 'error' => ''];
    
    try {
        error_log("Starting sendTxtFileAsNormalMessageNBR for: " . $file['name']);
        
        $max_txt_size = 2 * 1024 * 1024; 
        if (isset($file['size']) && $file['size'] > $max_txt_size) {
            error_log("File too large: " . $file['size'] . " bytes");
            return sendFileLink($chat_id, $file, "File txt terlalu besar untuk dikirim sebagai teks");
        }
        
        error_log("Downloading file content using multiple methods...");
        $txt_content = downloadFileWithMultipleMethodsNBR($file['id']);
        
        if ($txt_content === false || $txt_content === null) {
            error_log("Downloaded content is false or null");
            return sendFileLink($chat_id, $file, "Tidak dapat mengunduh file txt");
        }
        
        if (empty(trim($txt_content))) {
            error_log("Downloaded content is empty after trim");
            return sendFileLink($chat_id, $file, "File txt kosong");
        }
        
        error_log("Downloaded content length: " . strlen($txt_content));
        error_log("Content preview: " . substr($txt_content, 0, 100) . "...");
        
        $txt_content = fixEncodingAndCleanContentNBR($txt_content);
        
        if (empty(trim($txt_content))) {
            error_log("Content is empty after encoding fix and cleaning");
            return sendFileLink($chat_id, $file, "File txt kosong setelah pembersihan");
        }
        
        error_log("Content after encoding fix - length: " . strlen($txt_content));
        
        $max_message_length = 4000;
        
        $file_name = htmlspecialchars($file['name'], ENT_QUOTES, 'UTF-8');
        $message_header = "📄 <b>{$file_name}</b>\n\n";
        
        if (strlen($txt_content) > ($max_message_length - strlen($message_header))) {
            error_log("Content too long, splitting into chunks");
            return sendLongNormalTextContentNBR($chat_id, $file_name, $txt_content, $message_header);
        } else {
            $full_message = $message_header . $txt_content;
            error_log("Sending single message with length: " . strlen($full_message));
            
            if (!mb_check_encoding($full_message, 'UTF-8')) {
                error_log("Final message is not valid UTF-8, attempting to fix");
                $full_message = mb_convert_encoding($full_message, 'UTF-8', 'UTF-8');
            }
            
            $send_result = sendMessage($chat_id, $full_message);
            
            if ($send_result && isset($send_result['ok']) && $send_result['ok']) {
                $result['success'] = true;
                error_log("Successfully sent txt content as normal message");
            } else {
                error_log("Failed to send message. Result: " . print_r($send_result, true));
                return sendFileLink($chat_id, $file, "Gagal mengirim konten txt sebagai pesan");
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Exception in sendTxtFileAsNormalMessageNBR: " . $e->getMessage());
        return sendFileLink($chat_id, $file, "Error saat memproses file txt: " . $e->getMessage());
    }
}

function fixEncodingAndCleanContentNBR($content) {
    error_log("Starting encoding fix and content cleaning for NBR");
    
    $detected_encoding = mb_detect_encoding($content, ['UTF-8', 'Windows-1252', 'ISO-8859-1', 'ASCII'], true);
    error_log("Detected encoding: " . ($detected_encoding ?: 'unknown'));
    
    if ($detected_encoding && $detected_encoding !== 'UTF-8') {
        $content = mb_convert_encoding($content, 'UTF-8', $detected_encoding);
        error_log("Converted from $detected_encoding to UTF-8");
    } else if (!$detected_encoding || !mb_check_encoding($content, 'UTF-8')) {
        error_log("Encoding detection failed or invalid UTF-8, trying alternative approaches");
        
        $content = @mb_convert_encoding($content, 'UTF-8', 'Windows-1252');
        
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @iconv('UTF-8', 'UTF-8//IGNORE', $content);
        }
        
        if (!mb_check_encoding($content, 'UTF-8')) {
            $content = @mb_convert_encoding($content, 'UTF-8', 'UTF-8');
        }
    }
    
    $content = preg_replace('/\r\n|\r/', "\n", $content);
    $content = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $content);
    
    $content = preg_replace('/[\x{200B}-\x{200D}\x{FEFF}]/u', '', $content);
    $content = preg_replace('/[\x{00A0}]/u', ' ', $content);
    
    $content = trim($content);
    $content = preg_replace('/\n{3,}/', "\n\n", $content);
    $content = preg_replace('/[ \t]+/', ' ', $content);
    
    if (!mb_check_encoding($content, 'UTF-8')) {
        error_log("Content still not valid UTF-8 after all fixes, using fallback");
        $content = preg_replace_callback('/[^\x20-\x7E\n]/', function($matches) {
            return '?';
        }, $content);
    }
    
    error_log("Encoding fix completed, final content length: " . strlen($content));
    error_log("Is valid UTF-8: " . (mb_check_encoding($content, 'UTF-8') ? 'yes' : 'no'));
    
    return $content;
}

function downloadFileWithMultipleMethodsNBR($file_id) {
    error_log("Attempting to download NBR file with ID: $file_id");
    
    $file_content = downloadFileMethod1($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded NBR file with method1, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method1 failed for NBR file, trying method3");
    
    $file_content = downloadFileMethod3($file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded NBR file with method3, size: " . strlen($file_content));
        return $file_content;
    }
    error_log("Method3 failed for NBR file, trying direct URL");
    
    $file_content = downloadDirectUrl("https://drive.google.com/uc?export=download&id=" . $file_id);
    if ($file_content !== false && $file_content !== null && $file_content !== '') {
        error_log("Successfully downloaded NBR file with direct URL, size: " . strlen($file_content));
        return $file_content;
    }
    
    error_log("All download methods failed for NBR file ID: $file_id");
    return false;
}

function sendLongNormalTextContentNBR($chat_id, $file_name, $txt_content, $message_header) {
    $max_chunk_size = 3900; // Batasi per chunk
    $chunks = str_split($txt_content, $max_chunk_size);
    $total_chunks = count($chunks);
    
    $result = ['success' => true, 'error' => ''];
    
    error_log("Splitting NBR content into $total_chunks chunks");
    
    foreach ($chunks as $index => $chunk) {
        $chunk_number = $index + 1;
        $chunk_header = $message_header;
        
        if ($total_chunks > 1) {
            $chunk_header .= "📋 <b>Bagian {$chunk_number}/{$total_chunks}</b>\n\n";
        }
        
        // Kirim sebagai teks biasa tanpa tag <pre>
        $chunk_message = $chunk_header . $chunk;
        
        // Pastikan setiap chunk juga valid UTF-8
        if (!mb_check_encoding($chunk_message, 'UTF-8')) {
            $chunk_message = mb_convert_encoding($chunk_message, 'UTF-8', 'UTF-8');
        }
        
        error_log("Sending NBR chunk $chunk_number/$total_chunks with length: " . strlen($chunk_message));
        $send_result = sendMessage($chat_id, $chunk_message);
        
        if (!$send_result || !isset($send_result['ok']) || !$send_result['ok']) {
            $result['success'] = false;
            $result['error'] = "Gagal mengirim bagian {$chunk_number}";
            error_log("Failed to send NBR chunk $chunk_number. Result: " . print_r($send_result, true));
            break;
        }
        
        // Delay antar chunk
        if ($index < count($chunks) - 1) {
            sleep(1);
        }
    }
    
    return $result;
}

// Note: Fungsi-fungsi berikut sudah ada di nkl.php dan akan digunakan:
// - downloadFileMethod1()
// - downloadFileMethod3() 
// - downloadDirectUrl()
// - sendFileFromGoogleDrive()
// - sendFileLink()
// - getFilesFromGoogleDrive()

?>