<?php
require_once 'config.php';

function handleBuatToken($telegram_id, $chat_id, $text) {
    $user = isUserRegistered($telegram_id);
    if (!$user || $user['role'] !== 'Admin') {
        sendMessage($chat_id, "❌ Fitur ini hanya tersedia untuk Admin.");
        return;
    }

    $message = "🔑 <b>Buat Token Akses</b>\n\n";
    $message .= "Mau folder mana nih yang mau kamu buat tokennya?";
    
    $inline_keyboard = [
        [
            ['text' => '📁 NKL', 'callback_data' => 'create_token_NKL'],
            ['text' => '📁 NBR', 'callback_data' => 'create_token_NBR']
        ],
        [
            ['text' => '📋 Detail NKL', 'callback_data' => 'create_token_Detail NKL'],
            ['text' => '📋 Detail NBR', 'callback_data' => 'create_token_Detail NBR']
        ],
        [
            ['text' => '📈 TREND', 'callback_data' => 'create_token_TREND']
        ],
        [
            ['text' => '16 Toko Fokus NBR', 'callback_data' => 'create_token_16 Toko Fokus NBR']
        ],
        [
            ['text' => '❌ Batal', 'callback_data' => 'cancel_create_token']
        ]
    ];
    
    $reply_markup = [
        'inline_keyboard' => $inline_keyboard
    ];
    
    sendMessage($chat_id, $message, $reply_markup);
}

function handleCreateTokenCallback($telegram_id, $chat_id, $callback_data) {
    $user = isUserRegistered($telegram_id);
    if (!$user || $user['role'] !== 'Admin') {
        sendMessage($chat_id, "❌ Akses ditolak. Hanya admin yang dapat membuat token.");
        return;
    }
    
    if ($callback_data === 'cancel_create_token') {
        sendMessage($chat_id, "❌ Pembuatan token dibatalkan.");
        return;
    }
    
    $feature = str_replace('create_token_', '', $callback_data);

    $token = generateUniqueToken();
    
    $expires_at = date('Y-m-d H:i:s', strtotime('+24 hours'));
    
    try {
        $db = getDB();
        $stmt = $db->prepare("INSERT INTO tokens (token, feature, created_by, expires_at, created_at) VALUES (?, ?, ?, ?, ?)");
        $result = $stmt->execute([$token, $feature, $telegram_id, $expires_at, date('Y-m-d H:i:s')]);
        
        if ($result) {
            $message = "✅ <b>Token Berhasil Dibuat!</b>\n\n";
            $message .= "🔑 <b>Token:</b> <code>{$token}</code>\n";
            $message .= "📁 <b>Akses ke:</b> {$feature}\n";
            $message .= "⏰ <b>Berlaku hingga:</b> " . formatDateTime($expires_at) . "\n";
            $message .= "📋 <b>Status:</b> Belum digunakan\n\n";
            $message .= "💡 <i>Token ini dapat digunakan oleh pengguna yang tidak terdaftar untuk mengakses fitur {$feature} selama 24 jam.</i>\n\n";
            $message .= "📤 <i>Bagikan token ini kepada pengguna yang membutuhkan akses sementara.</i>";
            
            sendMessage($chat_id, $message);
            
            error_log("Token created: $token for feature: $feature by admin: $telegram_id");
        } else {
            sendMessage($chat_id, "❌ Gagal membuat token. Silakan coba lagi.");
        }
    } catch (Exception $e) {
        error_log("Error creating token: " . $e->getMessage());
        sendMessage($chat_id, "❌ Terjadi kesalahan saat membuat token: " . $e->getMessage());
    }
}

function generateUniqueToken($length = 10) {
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $max_attempts = 50;
    $attempt = 0;
    
    do {
        $attempt++;
        $token = '';
        for ($i = 0; $i < $length; $i++) {
            $token .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        try {
            $db = getDB();
            $stmt = $db->prepare("SELECT COUNT(*) FROM tokens WHERE token = ?");
            $stmt->execute([$token]);
            $exists = $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            error_log("Error checking token uniqueness: " . $e->getMessage());
            $exists = false; // Jika ada error, anggap token unique
        }
        
        if ($attempt >= $max_attempts) {
            throw new Exception("Unable to generate unique token after $max_attempts attempts");
        }
        
    } while ($exists);
    
    return $token;
}

function validateToken($token) {
    try {
        $db = getDB();
        
        $current_time = date('Y-m-d H:i:s');
        
        $stmt = $db->prepare("
            SELECT t.*, u.nama_depan, u.nama_belakang 
            FROM tokens t 
            LEFT JOIN users u ON t.created_by = u.telegram_id 
            WHERE t.token = ? AND t.expires_at > ? AND t.is_used = 0
        ");
        $stmt->execute([$token, $current_time]);
        $token_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        error_log("Token validation - Token: $token, Current time: $current_time");
        if ($token_data) {
            error_log("Token found - Expires at: " . $token_data['expires_at'] . ", Is used: " . $token_data['is_used']);
        } else {
            $stmt_debug = $db->prepare("SELECT token, expires_at, is_used, created_at FROM tokens WHERE token = ?");
            $stmt_debug->execute([$token]);
            $debug_data = $stmt_debug->fetch(PDO::FETCH_ASSOC);
            if ($debug_data) {
                error_log("Debug - Token exists but invalid: " . json_encode($debug_data));
            } else {
                error_log("Debug - Token does not exist in database");
            }
        }
        
        return $token_data ? $token_data : false;
    } catch (Exception $e) {
        error_log("Error validating token: " . $e->getMessage());
        return false;
    }
}

function useToken($token, $user_telegram_id) {
    try {
        $db = getDB();
        $current_time = date('Y-m-d H:i:s');
        
        $stmt = $db->prepare("
            UPDATE tokens 
            SET is_used = 1, 
                used_by = NULL, 
                used_at = ?, 
                used_by_info = ?
            WHERE token = ? AND is_used = 0 AND expires_at > ?
        ");
        
        $user_info = "unregistered_" . $user_telegram_id;
        $result = $stmt->execute([$current_time, $user_info, $token, $current_time]);
        
        $affected_rows = $stmt->rowCount();
        error_log("Token usage - Token: $token, User: $user_telegram_id, Affected rows: $affected_rows");
        
        if ($affected_rows > 0) {
            error_log("Token successfully marked as used: $token");
            return true;
        } else {
            error_log("No rows affected when marking token as used: $token");
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Error using token: " . $e->getMessage());
        
        try {
            $db = getDB();
            $current_time = date('Y-m-d H:i:s');
            
            $stmt = $db->prepare("
                UPDATE tokens 
                SET is_used = 1, used_by = NULL, used_at = ? 
                WHERE token = ? AND is_used = 0 AND expires_at > ?
            ");
            $result = $stmt->execute([$current_time, $token, $current_time]);
            
            $affected_rows = $stmt->rowCount();
            error_log("Fallback token usage - Affected rows: $affected_rows");
            
            return $result && $affected_rows > 0;
            
        } catch (Exception $e2) {
            error_log("Fallback token update also failed: " . $e2->getMessage());
            return false;
        }
    }
}

function handleTokenInput($telegram_id, $chat_id, $token) {
    $normalized_token = strtoupper(trim($token));
    
    error_log("Token input received - User: $telegram_id, Original: '$token', Normalized: '$normalized_token'");
    
    if (!preg_match('/^[A-Z0-9]{8,12}$/', $normalized_token)) {
        error_log("Invalid token format: $normalized_token");
        sendMessage($chat_id, "❌ Format token tidak valid. Token harus terdiri dari 8-12 karakter huruf besar dan angka.");
        return;
    }
    
    $token_data = validateToken($normalized_token);
    
    if (!$token_data) {
        error_log("Token validation failed for: $normalized_token");
        sendMessage($chat_id, "❌ Token tidak valid, sudah digunakan, atau sudah kadaluarsa. Silakan periksa kembali token Anda atau hubungi admin untuk mendapatkan token baru.");
        return;
    }
    
    error_log("Token validated successfully: " . json_encode([
        'token' => $token_data['token'],
        'feature' => $token_data['feature'],
        'expires_at' => $token_data['expires_at']
    ]));
    
    if (useToken($normalized_token, $telegram_id)) {
        $feature = $token_data['feature'];
        $admin_name = trim($token_data['nama_depan'] . ' ' . $token_data['nama_belakang']);
        
        $message = "✅ <b>Token Valid!</b>\n\n";
        $message .= "🔑 Anda mendapatkan akses sementara ke fitur: <b>{$feature}</b>\n";
        $message .= "👤 Token dibuat oleh: {$admin_name}\n";
        $message .= "⏰ Token akan segera diproses...\n\n";
        $message .= "📁 Memulai pengambilan data {$feature}...";
        
        sendMessage($chat_id, $message);
        
        executeTokenFeature($telegram_id, $chat_id, $feature);
        
        error_log("Token {$normalized_token} used successfully by {$telegram_id} for feature {$feature}");
    } else {
        error_log("Failed to mark token as used: $normalized_token");
        sendMessage($chat_id, "❌ Gagal menggunakan token. Token mungkin sudah digunakan atau terjadi kesalahan sistem. Silakan coba lagi atau hubungi admin.");
    }
}

function executeTokenFeature($telegram_id, $chat_id, $feature) {
    try {
        switch ($feature) {
            case 'NKL':
                if (file_exists('nkl.php')) {
                    require_once 'nkl.php';
                    handleNKL($telegram_id, $chat_id, 'NKL', true);
                } else {
                    sendMessage($chat_id, "⛔ File nkl.php tidak ditemukan.");
                }
                break;
                
            case 'NBR':
                if (file_exists('nbr.php')) {
                    require_once 'nbr.php';
                    handleNBR($telegram_id, $chat_id, 'NBR', true);
                } else {
                    sendMessage($chat_id, "⛔ File nbr.php tidak ditemukan.");
                }
                break;
                
            case 'Detail NKL':
                if (file_exists('detail-nkl.php')) {
                    require_once 'detail-nkl.php';
                    handleDetailNKL($telegram_id, $chat_id, 'Detail NKL', true);
                } else {
                    sendMessage($chat_id, "⛔ File detail-nkl.php tidak ditemukan.");
                }
                break;
                
            case 'Detail NBR':
                if (file_exists('detail-nbr.php')) {
                    require_once 'detail-nbr.php';
                    handleDetailNBR($telegram_id, $chat_id, 'Detail NBR', true);
                } else {
                    sendMessage($chat_id, "⛔ File detail-nbr.php tidak ditemukan.");
                }
                break;
                
            case 'TREND':
                if (file_exists('trend.php')) {
                    require_once 'trend.php';
                    handleTREND($telegram_id, $chat_id, 'TREND', true);
                } else {
                    sendMessage($chat_id, "⛔ File trend.php tidak ditemukan.");
                }
                break;
                
            case '16 Toko Fokus NBR':
                if (file_exists('16-toko-fokus-nbr.php')) {
                    require_once '16-toko-fokus-nbr.php';
                    handle16TokoFokus($telegram_id, $chat_id, '16 Toko Fokus NBR', true);
                } else {
                    sendMessage($chat_id, "⛔ File 16-toko-fokus-nbr.php tidak ditemukan.");
                }
                break;
                
            default:
                sendMessage($chat_id, "⛔ Feature tidak dikenal: {$feature}");
                error_log("Unknown feature requested: $feature");
                break;
        }
    } catch (Exception $e) {
        error_log("Error executing token feature '$feature': " . $e->getMessage());
        sendMessage($chat_id, "⛔ Terjadi kesalahan saat menjalankan fitur {$feature}. Silakan hubungi admin.");
    }
}

function cleanupExpiredTokens() {
    try {
        $db = getDB();
        $current_time = date('Y-m-d H:i:s');
        
        $stmt = $db->prepare("DELETE FROM tokens WHERE expires_at < ?");
        $result = $stmt->execute([$current_time]);
        $deleted_count = $stmt->rowCount();
        
        if ($deleted_count > 0) {
            error_log("Cleaned up {$deleted_count} expired tokens");
        }
        
        return $deleted_count;
    } catch (Exception $e) {
        error_log("Error cleaning up expired tokens: " . $e->getMessage());
        return 0;
    }
}

function listActiveTokens($admin_telegram_id) {
    try {
        $db = getDB();
        $current_time = date('Y-m-d H:i:s');
        
        $stmt = $db->prepare("
            SELECT 
                token, 
                feature, 
                created_at, 
                expires_at, 
                is_used, 
                used_at,
                used_by_info,
                CASE 
                    WHEN expires_at < ? THEN 'Expired'
                    WHEN is_used = 1 THEN 'Used'
                    ELSE 'Active'
                END as status
            FROM tokens 
            WHERE created_by = ? 
            ORDER BY created_at DESC 
            LIMIT 20
        ");
        $stmt->execute([$current_time, $admin_telegram_id]);
        $tokens = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return $tokens;
    } catch (Exception $e) {
        error_log("Error listing tokens: " . $e->getMessage());
        return [];
    }
}

// Fungsi helper untuk debugging (dapat dipanggil dari webhook untuk troubleshooting)
function debugTokenStatus($token) {
    try {
        $db = getDB();
        $current_time = date('Y-m-d H:i:s');
        
        $stmt = $db->prepare("
            SELECT 
                token,
                feature,
                created_at,
                expires_at,
                is_used,
                used_at,
                used_by_info,
                TIMESTAMPDIFF(HOUR, ?, expires_at) as hours_remaining,
                CASE 
                    WHEN expires_at < ? THEN 'EXPIRED'
                    WHEN is_used = 1 THEN 'USED'
                    ELSE 'ACTIVE'
                END as status
            FROM tokens 
            WHERE token = ?
        ");
        $stmt->execute([$current_time, $current_time, strtoupper($token)]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            error_log("DEBUG TOKEN STATUS: " . json_encode($result));
            return $result;
        } else {
            error_log("DEBUG: Token not found in database: " . strtoupper($token));
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Error debugging token: " . $e->getMessage());
        return false;
    }
}
?>